unit DataBindingReadU;

{
  Movie-watchers. A demonstration of XML document processing
  using the XML Data Binding features of Delphi 6 and 7.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 12 November 2001.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls, ExtCtrls, Grids, XMLIntf, DataBindingObjs, StrUtils;

type
  { The user interface }
  TfrmMovieWatchers = class(TForm)
    tabNavigation: TTabControl;
    lbxNavigation: TListBox;
    pgcDetails: TPageControl;
      tshMovie: TTabSheet;
        Label1: TLabel;
        edtTitle: TEdit;
        Label2: TLabel;
        edtRating: TEdit;
        Label3: TLabel;
        edtLength: TEdit;
        Label4: TLabel;
        edtDirector: TEdit;
        Label5: TLabel;
        lbxStars: TListBox;
        Label6: TLabel;
        memSynopsis: TMemo;
        Label7: TLabel;
        lbxCinemas: TListBox;
      tshCinema: TTabSheet;
        Label8: TLabel;
        edtName: TEdit;
        Label9: TLabel;
        edtPhone: TEdit;
        Label10: TLabel;
        edtAddress: TEdit;
        Label11: TLabel;
        memDirections: TMemo;
        cbxDisabledAccess: TCheckBox;
        cbxCandyBar: TCheckBox;
        Label13: TLabel;
        stgPricing: TStringGrid;
        Label12: TLabel;
        lbxMovies: TListBox;
      tshScreening: TTabSheet;
        Label14: TLabel;
        edtMovie: TEdit;
        Label15: TLabel;
        edtCinema: TEdit;
        Label16: TLabel;
        edtDates: TEdit;
        Label17: TLabel;
        edtSound: TEdit;
        cbxNoPasses: TCheckBox;
        Label18: TLabel;
        lbxSessions: TListBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure tabNavigationChange(Sender: TObject);
    procedure lbxNavigationClick(Sender: TObject);
    procedure lbxCinemasDblClick(Sender: TObject);
    procedure lbxCinemasKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure lbxMoviesDblClick(Sender: TObject);
    procedure lbxMoviesKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edtMovieDblClick(Sender: TObject);
    procedure edtMovieKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edtCinemaDblClick(Sender: TObject);
    procedure edtCinemaKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure lbxSessionsDblClick(Sender: TObject);
    procedure lbxSessionsKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    FMWObjs: IXMLMovieWatcherType;
    FMovies: TStringList;
    FCinemas: TStringList;
    FScreenings: TStringList;
    function FindCinema(Id: string): IXMLCinemaType;
    function FindMovie(Id: string): IXMLMovieType;
    function FindPrices(Id: string): IXMLPricesType;
    procedure LoadDocument(URI: string;
      MoviesList, CinemasList, ScreeningsList: TStringList);
    procedure ShowList(ListType: Integer; Name: string);
    procedure ShowMovie(Movie: IXMLMovieType);
    procedure ShowCinema(Cinema: IXMLCinemaType);
    procedure ShowScreening(Screening: IXMLScreeningType);
  public
  end;

var
  frmMovieWatchers: TfrmMovieWatchers;

implementation

{$R *.DFM}

resourcestring
  { Tab names }
  MoviesHdg       = '&Movies';
  CinemasHdg      = '&Cinemas';
  ScreeningsHdg   = '&Screenings';
  { Pricing headings }
  NameHdg         = 'Name';
  PeriodHdg       = 'Period';
  AdultHdg        = 'Adult';
  ChildHdg        = 'Child';
  DiscountHdg     = 'Discount';
  { Miscellaneous }
  InvalidDocument = 'Invalid movie-watcher XML document loaded'#13;
  ScreeningDesc   = '%s at %s';

const
  { Tab indexes }
  MoviesTab     = 0;
  CinemasTab    = 1;
  ScreeningsTab = 2;
  { Miscellaneous }
  TimeFormat    = 'h:nn';
  DollarsFormat = '%6.2f';
  Tab           = #09;

type
  { Provide an object wrapper for an interface }
  TInterfaceWrapper = class(TObject)
  private
    FIntf: IUnknown;
  public
    constructor Create(Intf: IUnknown);
    property Intf: IUnknown read FIntf write FIntf;
  end;

constructor TInterfaceWrapper.Create(Intf: IUnknown);
begin
  inherited Create;
  FIntf := Intf;
end;

{ TfrmMovieWatchers -----------------------------------------------------------}

{ Initialisation - load XML document and process }
procedure TfrmMovieWatchers.FormCreate(Sender: TObject);
begin
  tabNavigation.Tabs[0] := MoviesHdg;
  tabNavigation.Tabs[1] := CinemasHdg;
  tabNavigation.Tabs[2] := ScreeningsHdg;
  with stgPricing do
  begin
    ColWidths[0] := 50;
    ColWidths[1] := 83;
    ColWidths[2] := 40;
    ColWidths[3] := 40;
    ColWidths[4] := 40;
    Cells[0, 0]  := NameHdg;
    Cells[1, 0]  := PeriodHdg;
    Cells[2, 0]  := AdultHdg;
    Cells[3, 0]  := ChildHdg;
    Cells[4, 0]  := DiscountHdg;
  end;
  { Create the internal lists }
  FMovies     := TStringList.Create;
  FCinemas    := TStringList.Create;
  FScreenings := TStringList.Create;
  { Open the XML file }
  try
    LoadDocument(IfThen(ParamCount > 0, ParamStr(1), 'movie-watcher.xml'),
      FMovies, FCinemas, FScreenings);
  except on Error: Exception do
    begin
      MessageDlg(Error.Message, mtError, [mbOK], 0);
      Halt;
    end;
  end;
  { Start out on the movies }
  ShowList(MoviesTab, '');
end;

{ Release resources }
procedure TfrmMovieWatchers.FormDestroy(Sender: TObject);
var
  Index: Integer;
begin
  FMWObjs := nil;
  for Index := 0 to FMovies.Count - 1 do
    FMovies.Objects[Index].Free;
  FMovies.Free;
  for Index := 0 to FCinemas.Count - 1 do
    FCinemas.Objects[Index].Free;
  FCinemas.Free;
  for Index := 0 to FScreenings.Count - 1 do
    FScreenings.Objects[Index].Free;
  FScreenings.Free;
end;

{ Find a cinema given its id }
function TfrmMovieWatchers.FindCinema(Id: string): IXMLCinemaType;
var
  Index: Integer;
begin
  for Index := 0 to FMWObjs.Cinemas.Count - 1 do
    if FMWObjs.Cinemas[Index].Id = Id then
    begin
      Result := FMWObjs.Cinemas[Index];
      Exit;
    end;
  Result := nil;
end;

{ Find a movie given its id }
function TfrmMovieWatchers.FindMovie(Id: string): IXMLMovieType;
var
  Index: Integer;
begin
  for Index := 0 to FMWObjs.Movies.Count - 1 do
    if FMWObjs.Movies[Index].Id = Id then
    begin
      Result := FMWObjs.Movies[Index];
      Exit;
    end;
  Result := nil;
end;

{ Find a pricing scheme given its id }
function TfrmMovieWatchers.FindPrices(Id: string): IXMLPricesType;
var
  Index1, Index2: Integer;
begin
  for Index1 := 0 to FMWObjs.Cinemas.Count - 1 do
    for Index2 := 0 to FMWObjs.Cinemas[Index1].Pricing.Count - 1 do
      if FMWObjs.Cinemas[Index1].Pricing[Index2].Id = Id then
      begin
        Result := FMWObjs.Cinemas[Index1].Pricing[Index2];
        Exit;
      end;
  Result := nil;
end;

{ Load XML document and process into string lists
  with references to the appropriate objects }
procedure TfrmMovieWatchers.LoadDocument(URI: string;
  MoviesList, CinemasList, ScreeningsList: TStringList);
var
  Index: Integer;
begin
  { Create the XML parser }
  FMWObjs := LoadMovieWatcher(URI);
  with FMWObjs do
  begin
    { Are they all here? }
    if (Movies.Count = 0) or (Cinemas.Count = 0) or
        (Screenings.Count = 0) then
      raise Exception.Create(InvalidDocument + URI);
    { Step through the object's lists and convert to output format }
    for Index := 0 to Movies.Count - 1  do
      MoviesList.AddObject(Movies[Index].Name,
        TInterfaceWrapper.Create(Movies[Index]));
    for Index := 0 to Cinemas.Count - 1 do
      CinemasList.AddObject(Cinemas[Index].Name,
        TInterfaceWrapper.Create(Cinemas[Index]));
    for Index := 0 to Screenings.Count - 1 do
      ScreeningsList.AddObject(Format(ScreeningDesc,
        [FindMovie(Screenings[Index].MovieId).Name,
        FindCinema(Screenings[Index].CinemaId).Name]),
        TInterfaceWrapper.Create(Screenings[Index]));
  end;
end;

{ Show list of items and select one }
procedure TfrmMovieWatchers.ShowList(ListType: Integer; Name: string);
begin
  tabNavigation.TabIndex  := ListType;
  tabNavigationChange(tabNavigation);
  lbxNavigation.ItemIndex := lbxNavigation.Items.IndexOf(Name);
  if lbxNavigation.ItemIndex = -1 then
    lbxNavigation.ItemIndex := 0;
  lbxNavigationClick(lbxNavigation);
end;

{ Show selected details in listbox }
procedure TfrmMovieWatchers.tabNavigationChange(Sender: TObject);
begin
  with lbxNavigation do
  begin
    Items.BeginUpdate;
    Items.Clear;
    if tabNavigation.TabIndex = MoviesTab then
      Items := FMovies
    else if tabNavigation.TabIndex = CinemasTab then
      Items := FCinemas
    else if tabNavigation.TabIndex = ScreeningsTab then
      Items := FScreenings;
    Items.EndUpdate;
  end;
  lbxNavigation.ItemIndex := 0;
  lbxNavigationClick(lbxNavigation);
  ActiveControl           := lbxNavigation;
end;

{ Display details for a movie }
procedure TfrmMovieWatchers.ShowMovie(Movie: IXMLMovieType);
var
  Index: Integer;
begin
  with Movie do
  begin
    edtTitle.Text          := Name;
    edtRating.Text         := Rating;
    edtLength.Text         := IntToStr(Length);
    edtDirector.Text       := Director;
    memSynopsis.Lines.Text := Synopsis;
    with lbxStars do
    begin
      Clear;
      for Index := 0 to Starring.Count - 1 do
        Items.Add(Starring.Star[Index]);
    end;
    { Show which cinemas it is playing at }
    with lbxCinemas.Items do
    begin
      BeginUpdate;
      Clear;
      for Index := 0 to FScreenings.Count - 1 do
        if (TInterfaceWrapper(FScreenings.Objects[Index]).Intf as
            IXMLScreeningType).MovieId = Id then
          AddObject(FindCinema((TInterfaceWrapper(FScreenings.Objects[Index]).
            Intf as IXMLScreeningType).CinemaId).Name,
            FScreenings.Objects[Index]);
      if Count > 0 then
        lbxCinemas.ItemIndex := 0;
      EndUpdate;
    end;
  end;
  pgcDetails.ActivePage := tshMovie;
end;

{ Display details for a cinema }
procedure TfrmMovieWatchers.ShowCinema(Cinema: IXMLCinemaType);
var
  Index: Integer;
begin
  with Cinema do
  begin
    edtName.Text              := Name;
    edtPhone.Text             := Phone;
    edtAddress.Text           := Address;
    memDirections.Lines.Text  := Directions;
    cbxCandyBar.Checked       := Facilities.CandyBar;
    cbxDisabledAccess.Checked := Facilities.DisabledAccess;
    with stgPricing do
    begin
      RowCount := Pricing.Count + 1;
      for Index := 0 to Pricing.Count - 1 do
        with Pricing.Prices[Index] do
        begin
          Cells[0, Index + 1] := Name;
          Cells[1, Index + 1] := Period;
          Cells[2, Index + 1] := Format(DollarsFormat, [Adult]);
          Cells[3, Index + 1] := Format(DollarsFormat, [Child]);
          Cells[4, Index + 1] :=
            IfThen(Discount = 0, '', Format(DollarsFormat, [Discount]));
        end;
    end;
    { Show which movies it screens }
    with lbxMovies.Items do
    begin
      BeginUpdate;
      Clear;
      for Index := 0 to FScreenings.Count - 1 do
        if (TInterfaceWrapper(FScreenings.Objects[Index]).Intf as
            IXMLScreeningType).CinemaId = Id then
          AddObject(FindMovie((TInterfaceWrapper(FScreenings.Objects[Index]).
            Intf as IXMLScreeningType).MovieId).Name,
            FScreenings.Objects[Index]);
      if Count > 0 then
        lbxMovies.ItemIndex := 0;
      EndUpdate;
    end;
  end;
  pgcDetails.ActivePage := tshCinema;
end;

{ Display details for a screening }
procedure TfrmMovieWatchers.ShowScreening(Screening: IXMLScreeningType);
var
  Index: Integer;
begin
  with Screening do
  begin
    edtMovie.Text       := FindMovie(MovieId).Name;
    edtCinema.Text      := FindCinema(CinemaId).Name;
    edtDates.Text       := DateToStr(StartDate) + ' - ' + DateToStr(EndDate);
    edtSound.Text       := Features.DigitalSound;
    cbxNoPasses.Checked := Restrictions.NoPasses;
    with lbxSessions.Items do
    begin
      BeginUpdate;
      Clear;
      for Index := 0 to Sessions.Count - 1 do
        Add(Sessions.Session[Index].Text + Tab +
          FindPrices(Sessions.Session[Index].PriceId).Name);
      if Count > 0 then
        lbxSessions.ItemIndex := 0;
      EndUpdate;
    end;
  end;
  pgcDetails.ActivePage := tshScreening;
end;

{ Select an item to display its details }
procedure TfrmMovieWatchers.lbxNavigationClick(Sender: TObject);
begin
  with lbxNavigation do
  begin
    if ItemIndex < 0 then
      ItemIndex := 0;
    if tabNavigation.TabIndex = MoviesTab then
      ShowMovie(TInterfaceWrapper(Items.Objects[ItemIndex]).Intf as
        IXMLMovieType)
    else if tabNavigation.TabIndex = CinemasTab then
      ShowCinema(TInterfaceWrapper(Items.Objects[ItemIndex]).Intf as
        IXMLCinemaType)
    else if tabNavigation.TabIndex = ScreeningsTab then
      ShowScreening(TInterfaceWrapper(Items.Objects[ItemIndex]).Intf as
        IXMLScreeningType);
  end;
end;

{ Go to the screening details for a movie }
procedure TfrmMovieWatchers.lbxCinemasDblClick(Sender: TObject);
begin
  ShowList(ScreeningsTab, Format(ScreeningDesc,
    [edtTitle.Text, lbxCinemas.Items[lbxCinemas.ItemIndex]]));
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.lbxCinemasKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxCinemasDblClick(lbxCinemas);
end;

{ Go to the screening details for a cinema }
procedure TfrmMovieWatchers.lbxMoviesDblClick(Sender: TObject);
begin
  ShowList(ScreeningsTab, Format(ScreeningDesc,
    [lbxMovies.Items[lbxMovies.ItemIndex], edtName.Text]));
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.lbxMoviesKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxMoviesDblClick(lbxMovies);
end;

{ Go to the movie details }
procedure TfrmMovieWatchers.edtMovieDblClick(Sender: TObject);
begin
  ShowList(MoviesTab, edtMovie.Text);
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.edtMovieKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_RETURN then
    edtMovieDblClick(edtMovie);
end;

{ Go to the cinema details }
procedure TfrmMovieWatchers.edtCinemaDblClick(Sender: TObject);
begin
  ShowList(CinemasTab, edtCinema.Text);
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.edtCinemaKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    edtCinemaDblClick(edtCinema);
end;

{ Go to the cinema/pricing details }
procedure TfrmMovieWatchers.lbxSessionsDblClick(Sender: TObject);
var
  Index: Integer;
  Pricing: string;
begin
  with lbxSessions do
  begin
    Index   := Pos(Tab, Items[ItemIndex]);
    Pricing := Copy(Items[ItemIndex], Index + 1, Length(Items[ItemIndex]));
  end;
  ShowList(CinemasTab, edtCinema.Text);
  with stgPricing do
    for Index := 1 to RowCount - 1 do
      if Cells[0, Index] = Pricing then
      begin
        Row := Index;
        Exit;
      end;
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.lbxSessionsKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxSessionsDblClick(lbxSessions);
end;

end.
